importScript('../misc/init.js')
;(function () {
  options.logSubTest('ftdi.js')

  const { powerController } = deviceManager.getSupportedPeripherals()
  if (powerController === null) {
    logger.info('SKIPPED: No PMU device, test skipped')
    return
  }
  if (powerController.manufacturer !== 'ardent') {
    logger.info('SKIPPED: Ardent device not found, test skipped')
    return
  }

  const logfile = 'ftdi_test.log'
  if (!options.get('logger_ftdi_created')) {
    logger.create('ftdi_test', logfile)
    options.set('logger_ftdi_created', true)
  }

  const unexpected = new Error("It isn't failed")
  const shared = {}

  const writeMessage = (log = false) => {
    const message = '\u0002pmu(measual)\u0003'
    const result = ftdi.write(shared.idx, Buffer.from(message))
    const errorCode = ftdi.getErrorCode(shared.idx)
    if (result <= 0) {
      throw new Error('Failed to write data to FTDI device')
    }
    if (log) {
      logger.info(
        `PASS: ftdi.write wrote message "${message}" to idx=${shared.idx}, errorCode: ${errorCode}, result: ${result}`
      )
    }
  }

  {
    const devices = ftdi.getDeviceList({
      vendor: 0x0403,
      product: 0x6001
    })
    const serialNumbers = devices.map(device => device.serial)
    logger.info(
      `PASS: ftdi.getDeviceList listed all FTDI devices, ${serialNumbers}`
    )

    if (devices.length === 0) {
      logger.info('SKIPPED: No PMU device, test skipped')
      return
    }

    const { port } = deviceManager.getPortInfo()
    shared.selectedDevice = devices.filter(device =>
      device.serial.endsWith(`-${port}`)
    )[0]
  }

  {
    const devices = ftdi.getDeviceList()
    const serialNumbers = devices.map(device => device.serial)
    logger.info(
      `PASS: ftdi.getDeviceList listed all FTDI devices when no options are provided, ${serialNumbers}`
    )

    if (devices.length === 0) {
      logger.info('SKIPPED: No PMU device, test skipped')
      return
    }
  }

  try {
    ftdi.open({
      vendor: 0x0403,
      product: 0x6001,
      serial: 'invalid serial number'
    })
    throw unexpected
  } catch (e) {
    if (e === unexpected) {
      throw new Error('ftdi.open did not fail with an invalid serial number')
    }
    logger.info(`PASS: Ftdi.open threw an expected error, ${e.message}`)
  }

  {
    const { serial } = shared.selectedDevice
    const idx = ftdi.open({
      vendor: 0x0403,
      product: 0x6001,
      serial
    })
    logger.info(
      `PASS: ftdi.open opened a FTDI device using serial ${serial}, idx=${idx}`
    )
    shared.idx = idx
  }

  ftdi.setBaudRate(shared.idx, 115200)
  logger.info(`PASS: ftdi.setBaudRate set baudrate, idx=${shared.idx}`)

  writeMessage()

  {
    utility.sleep(1000)
    const data = ftdi.read(shared.idx).toString()
    const errorCode = ftdi.getErrorCode(shared.idx)
    if (data.length === 0) {
      throw new Error('Expecting non-zero data')
    }
    logger.info(
      `PASS: ftdi.read read data from idx=${
        shared.idx
      }, errorCode: ${errorCode}, data: ${data.trim()}`
    )
  }

  ftdi.flushBuffer(shared.idx)
  logger.info(`PASS: ftdi.flushBuffer flushed buffer from idx=${shared.idx}`)

  {
    const buffered = true
    const data = ftdi.read(shared.idx, 0, buffered)
    if (data.length !== 0) {
      throw new Error('Expecting empty buffer after flush buffer')
    }
    logger.info(`PASS: ftdi.read read empty buffer from idx=${shared.idx}`)
  }

  try {
    ftdi.pipe(shared.idx, 'invalid descriptor name')
    throw unexpected
  } catch (err) {
    if (err === unexpected) {
      throw unexpected
    }
    logger.info(
      `PASS: ftdi.pipe threw an expected error when piped to an invalid log descriptor`
    )
  }

  {
    shared.logname = 'ftdi_test'
    const result = ftdi.pipe(shared.idx, shared.logname)
    const errorCode = ftdi.getErrorCode(shared.idx)
    if (!result) {
      throw new Error('Pipe failed')
    }
    logger.info(
      `PASS: ftdi.pipe piped FTDI device to ${shared.logname}, errorCode: ${errorCode}, result: ${result}`
    )
  }

  {
    const loggerDataBefore = utility.readFile(logfile)
    logger.info(shared.logname, 'Test log from script')
    const loggerDataAfter = utility.readFile(logfile)
    if (loggerDataBefore.length === loggerDataAfter.length) {
      throw new Error(
        'message sent to piped logger was not written in the file'
      )
    }
    logger.info(
      `PASS: ftdi.pipe piped data written to the logger into the piped log file`
    )
    shared.lastLoggerSize = loggerDataAfter.length
  }

  {
    writeMessage(false)
    utility.sleep(1000)
    const loggerDataAfter = utility.readFile(logfile)
    if (loggerDataAfter.length === shared.lastLoggerSize) {
      throw new Error(
        'message received after ftdi.write was not written in the log file'
      )
    }
    logger.info(
      `PASS: ftdi.pipe piped response from the device to the piped log file`
    )
    shared.lastLoggerSize = loggerDataAfter.length
  }

  {
    ftdi.useBinaryMode(shared.idx)
    writeMessage(false)
    utility.sleep(1000)
    const loggerDataAfter = utility.readFile(logfile)
    if (loggerDataAfter.length !== shared.lastLoggerSize) {
      throw new Error(
        'message received after ftdi.write was written in the log file under binary mode'
      )
    }
    logger.info(
      `PASS: ftdi.useBinaryMode disabled writing response data to the log file`
    )
    shared.lastLoggerSize = loggerDataAfter.length
  }

  {
    ftdi.useTextMode(shared.idx)
    writeMessage(false)
    utility.sleep(1000)
    const loggerDataAfter = utility.readFile(logfile)
    if (loggerDataAfter.length === shared.lastLoggerSize) {
      throw new Error(
        'message received after ftdi.write was not written in the log file under text mode'
      )
    }
    logger.info(
      `PASS: ftdi.useTextMode enabled writing response data to the log file`
    )
    shared.lastLoggerSize = loggerDataAfter.length
  }

  {
    ftdi.unpipe(shared.idx)
    writeMessage(false)
    utility.sleep(1000)
    const loggerDataAfter = utility.readFile(logfile)
    if (loggerDataAfter.length !== shared.lastLoggerSize) {
      throw new Error(
        'message received after ftdi.write was written in the log file after unpipe'
      )
    }
    logger.info(
      `PASS: ftdi.unpipe unpiped idx=${shared.idx} from the logger "${shared.logname}"`
    )
  }

  ftdi.close(shared.idx)
  logger.info(`PASS: ftdi.close closed idx=${shared.idx}`)

  try {
    ftdi.write(shared.idx, Buffer.from('\u0002pmu(measual)\u0003'))
    throw unexpected
  } catch (err) {
    if (err === unexpected) {
      throw unexpected
    }
    logger.info(
      `PASS: ftdi.write threw an expected error when a message is written to a closed device`
    )
  }

  try {
    ftdi.read(shared.idx).toString()
  } catch (err) {
    if (err === unexpected) {
      throw unexpected
    }
    logger.info(
      `PASS: ftdi.read threw an expected error when a message is read from a closed device`
    )
  }
})()
