importScript('../misc/init.js')
;(function () {
  options.logSubTest('ftdi_nagative.js')

  const { powerController } = deviceManager.getSupportedPeripherals()
  if (powerController === null) {
    logger.info('SKIPPED: No PMU device, test skipped')
    return
  }
  if (powerController.manufacturer !== 'ardent') {
    logger.info('SKIPPED: Ardent device not found, test skipped')
    return
  }

  function runNegativeTest (func, funcName, errorMessage, errorReason) {
    try {
      func()
      throw new Error('Exception Not Triggered')
    } catch (e) {
      if (e.message === 'Exception Not Triggered') {
        throw new Error(
          'expected an exception, but exception was not triggered'
        )
      }
      if (!e.message.match(errorMessage)) {
        throw new Error(
          `${funcName} threw an unexpected exception "${e.message}"`
        )
      }
      logger.info(`PASS: ${errorReason}.`)
      return e
    }
  }

  function testParamTypeError (func, funcName, param, type) {
    runNegativeTest(
      func,
      funcName,
      `${funcName} : argument "${param}" must be (a|an) ${type}`,
      `${funcName} threw an exception when it was called with an invalid argument for "${param}"`
    )
  }

  function testInvalidIndexError (func, funcName) {
    runNegativeTest(
      func,
      funcName,
      `${funcName} : invalid device index`,
      `${funcName} threw an exception when it was called with an uninitialized device index`
    )
  }

  const paramErrorTestCases = [
    {
      func: ftdi.close.bind(ftdi, 'asdf'),
      funcName: 'ftdi.close',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.flushBuffer.bind(ftdi, 'asdf'),
      funcName: 'ftdi.flushBuffer',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.flushBuffer.bind(ftdi, 0, 'asdf'),
      funcName: 'ftdi.flushBuffer',
      param: 'offset',
      type: 'Number'
    },
    {
      func: ftdi.getDeviceList.bind(ftdi, { vendor: '1231' }),
      funcName: 'ftdi.getDeviceList',
      param: 'options.vendor',
      type: 'Number'
    },
    {
      func: ftdi.getDeviceList.bind(ftdi, { product: '1231' }),
      funcName: 'ftdi.getDeviceList',
      param: 'options.product',
      type: 'Number'
    },
    {
      func: ftdi.getErrorCode.bind(ftdi, 'asdf'),
      funcName: 'ftdi.getErrorCode',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.open.bind(ftdi, { vendor: '1231' }),
      funcName: 'ftdi.open',
      param: 'deviceInfo.vendor',
      type: 'Number'
    },
    {
      func: ftdi.open.bind(ftdi, { vendor: 0x0403, product: '1231' }),
      funcName: 'ftdi.open',
      param: 'deviceInfo.product',
      type: 'Number'
    },
    {
      func: ftdi.open.bind(ftdi, {
        vendor: 0x0403,
        product: 0x6001,
        interface: '1231'
      }),
      funcName: 'ftdi.open',
      param: 'deviceInfo.interface',
      type: 'Number'
    },
    {
      func: ftdi.open.bind(ftdi, {
        vendor: 0x0403,
        product: 0x6001,
        interface: 1,
        description: 1
      }),
      funcName: 'ftdi.open',
      param: 'deviceInfo.description',
      type: 'String'
    },
    {
      func: ftdi.open.bind(ftdi, {
        vendor: 0x0403,
        product: 0x6001,
        interface: 1,
        serial: 1
      }),
      funcName: 'ftdi.open',
      param: 'deviceInfo.serial',
      type: 'String'
    },
    {
      func: ftdi.open.bind(ftdi, {
        vendor: 0x0403,
        product: 0x6001,
        interface: 1,
        readInterval: '1231'
      }),
      funcName: 'ftdi.open',
      param: 'deviceInfo.readInterval',
      type: 'Number'
    },
    {
      func: ftdi.pipe.bind(ftdi, 'asdf'),
      funcName: 'ftdi.pipe',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.pipe.bind(ftdi, 0, 0),
      funcName: 'ftdi.pipe',
      param: 'logname',
      type: 'String'
    },
    {
      func: ftdi.read.bind(ftdi, 'asdf'),
      funcName: 'ftdi.read',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.read.bind(ftdi, 0, 'asdf'),
      funcName: 'ftdi.read',
      param: 'size',
      type: 'Number'
    },
    {
      func: ftdi.read.bind(ftdi, 0, 0, 'asdf'),
      funcName: 'ftdi.read',
      param: 'buffered',
      type: 'Boolean'
    },
    {
      func: ftdi.setBaudRate.bind(ftdi, 'asdf'),
      funcName: 'ftdi.setBaudRate',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.setBaudRate.bind(ftdi, 0, 'asdf'),
      funcName: 'ftdi.setBaudRate',
      param: 'baudRate',
      type: 'Number'
    },
    {
      func: ftdi.setParity.bind(ftdi, 'asdf'),
      funcName: 'ftdi.setParity',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.setParity.bind(ftdi, 0, 1),
      funcName: 'ftdi.setParity',
      param: 'parity',
      type: 'String'
    },
    {
      func: ftdi.setBitmode.bind(ftdi, 'asdf', {}),
      funcName: 'ftdi.setBitmode',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.setBitmode.bind(ftdi, 0, { bitmask: 'asdf' }),
      funcName: 'ftdi.setBitmode',
      param: 'bitmode.bitmask',
      type: 'Number'
    },
    {
      func: ftdi.setBitmode.bind(ftdi, 0, { bitmask: 0, mode: 'asdf' }),
      funcName: 'ftdi.setBitmode',
      param: 'bitmode.mode',
      type: 'Number'
    },
    {
      func: ftdi.unpipe.bind(ftdi, 'asdf'),
      funcName: 'ftdi.unpipe',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.useBinaryMode.bind(ftdi, 'asdf'),
      funcName: 'ftdi.useBinaryMode',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.useTextMode.bind(ftdi, 'asdf'),
      funcName: 'ftdi.useTextMode',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.write.bind(ftdi, 'asdf'),
      funcName: 'ftdi.write',
      param: 'devIdx',
      type: 'Number'
    },
    {
      func: ftdi.write.bind(ftdi, 0, true),
      funcName: 'ftdi.write',
      param: 'data',
      type: 'Buffer'
    }
  ]

  for (const testCase of paramErrorTestCases) {
    const { func, funcName, param, type } = testCase
    testParamTypeError(func, funcName, param, type)
  }

  const indexErrorTestCases = [
    { func: ftdi.close.bind(ftdi, 0), funcName: 'ftdi.close' },
    {
      func: ftdi.write.bind(ftdi, 0, Buffer.from([1, 2, 3, 4])),
      funcName: 'ftdi.write'
    },
    { func: ftdi.flushBuffer.bind(ftdi, 0), funcName: 'ftdi.flushBuffer' },
    { func: ftdi.getErrorCode.bind(ftdi, 0), funcName: 'ftdi.getErrorCode' }
  ]

  for (const testCase of indexErrorTestCases) {
    const { func, funcName } = testCase
    testInvalidIndexError(func, funcName)
  }
})()
