importScript('/functional/misc/init.js')
importScript('/lib/nvme/index.js')
;(function () {
  options.logSubTest('namespace_management.js')

  nvme.probeController()
  logger.info('PASS: Successfully connected to NVMe controller')

  const { data: controllerData } = nvme.identifyController()
  logger.info('PASS: Successfully sent Identify Controller')

  const isSupported = controllerData.OACS & 0xf
  if (!isSupported) {
    logger.info('SKIP: Namespace Management is not supported.')
  }

  const { data: commonNamespace } = nvme.identifyNamespace(0xffffffff)
  logger.info(`PASS: Successfully retrieved common namespace info`)

  const { namespaces: allocatedNamespaces } = nvme.identifyAllocatedNamespaces()
  logger.info(
    `PASS: Successfully retrieved allocated namespaces [${allocatedNamespaces}]`
  )

  for (const nsid of allocatedNamespaces) {
    nvme.identifyAllocatedNamespace(nsid)
    logger.info(`PASS: Successfully retrieved allocated namespace ${nsid}`)
    nvme.namespaceManagement(1, { nsid })
    logger.info(`PASS: Successfully deleted allocated namespace ${nsid}`)
  }

  if (nvme.identifyAllocatedNamespaces().namespaces.length !== 0) {
    throw new Error(
      'Found an allocated namespace after deleting all namespaces.'
    )
  }
  logger.info('PASS: Confirmed there are no allocated namespaces')

  const flbas = 0
  const totalCap = Number(controllerData.TNVMCAP)
  const maxNamespaces = controllerData.NN
  const lbaSize = 1 << commonNamespace.LBAF[flbas].LBADS
  const nsSize = Math.floor(totalCap / maxNamespaces / lbaSize / 2)

  for (let loop = 0; loop < maxNamespaces; loop++) {
    const options = {}
    options.nsze = nsSize
    options.ncap = nsSize
    const { cqe } = nvme.namespaceManagement(0, options)
    const nsid = cqe.cdw
    logger.info(`PASS: Successfully allocated namespace ${nsid}`)
  }

  options.runNegativeTest(
    () => nvme.namespaceManagement(0, { nsze: nsSize, ncap: nsSize }),
    'command error',
    'when all available namespaces are exhausted'
  )

  for (let nsid = 1; nsid <= maxNamespaces; nsid++) {
    nvme.namespaceAttachment(0, nsid, [controllerData.CNTLID])
    logger.info(
      `PASS: Successfully attached the current controller to namespace ${nsid}`
    )
  }

  const { namespaces: activeNamespaces } = nvme.identifyActiveNamespaces()
  for (let nsid = 1; nsid <= maxNamespaces; nsid++) {
    if (activeNamespaces.includes(nsid)) {
      logger.info(`PASS: Confirmed namespace ${nsid} is active`)
    } else {
      throw new Error(`namespace ${nsid} is not active after it is attached `)
    }
  }

  for (let nsid = 1; nsid <= maxNamespaces; nsid++) {
    nvme.namespaceAttachment(1, nsid, [controllerData.CNTLID])
    logger.info(
      `PASS: Successfully detached the current controller from namespace ${nsid}`
    )
  }

  if (nvme.identifyActiveNamespaces().namespaces.length === 0) {
    logger.info(`PASS: Confirmed all namespaces are inactive`)
  } else {
    throw new Error(`Found active namespaces after detaching all namespaces`)
  }

  nvme.namespaceManagement(1)
  logger.info(`PASS: Successfully removed all namespaces`)
  if (nvme.identifyAllocatedNamespaces().namespaces.length === 0) {
    logger.info(`PASS: Confirmed all namespaces are removed`)
  } else {
    throw new Error(`Found allocated namespaces after removing all namespaces`)
  }

  const nsid = nvme.namespaceManagement(0, {
    nsze: totalCap / lbaSize,
    ncap: totalCap / lbaSize
  }).cqe.cdw
  logger.info(`PASS: Successfully created namespace ${nsid}`)

  nvme.namespaceAttachment(0, nsid, [controllerData.CNTLID])
  logger.info(`PASS: Successfully attached the controller to namespace ${nsid}`)

  if (nvme.identifyController().data.UNVMCAP === 0n) {
    logger.info(`PASS: Confirmed there is no unused capacity`)
  } else {
    throw new Error('Found an unused capacity')
  }
})()
