/* globals importScript, utility, logger, nvme, powerController, options */

'use strict'

importScript('misc/init.js')

function testPowerCycleUsingWorkloadGenerator () {
  const { workloadGenerator } = options.get('powercycle') || {}
  const { duration = 30, loops = 1 } = workloadGenerator || {}
  options.logSubTest('testPowerCycleUsingWorkloadGenerator', {
    duration,
    loops
  })
  nvme.startWorkloadGenerator({
    nsid: 1,
    queueDepth: 1024,
    xferSize: 128 * 1024,
    duration,
    loops,
    readPercent: 0,
    workload: 'random',
    features: {
      powerControl: {
        enabled: true,
        detectionTimeoutMs: 1000 * 30,
        removalTimeoutMs: 1000 * 30,
        offHoldTimeMs: 1000 * 10,
        onHoldTimeMs: 1000 * 30
      }
    }
  })
  logger.info(
    'PASS: Successfully ran workload generator with power control feature'
  )
  nvme.probeController()
  logger.info(
    'PASS: Successfully identified the drive after running workload generator'
  )
}

function testPowerCycleUsingWorkloadGeneratorAggresivePc () {
  const { workloadGenerator } = options.get('powercycle') || {}
  const { duration = 30, loops = 1 } = workloadGenerator || {}
  options.logSubTest('testPowerCycleUsingWorkloadGenerator', {
    duration,
    loops
  })

  const t0 = new Date()
  while (new Date() - t0 < duration * 1000) {
    const loopDuration = 30
    nvme.startWorkloadGenerator({
      nsid: 1,
      queueDepth: 1024,
      xferSize: 128 * 1024,
      duration: loopDuration,
      loops,
      readPercent: 0,
      workload: 'random',
      features: {
        powerControl: {
          enabled: true,
          detectionTimeoutMs: 1000 * 30,
          removalTimeoutMs: 1000 * 30,
          offHoldTimeMs: 1000 * 10,
          onHoldTimeMs: 1000 * 30
        }
      }
    })
    logger.info(
      'PASS: Successfully ran workload generator with power control feature'
    )
    nvme.probeController()
    logger.info(
      'PASS: Successfully identified the drive after running workload generator'
    )
  }
}

function testShutdownWithReset () {
  options.logSubTest('testShutdownWithReset')

  powerController.init()
  nvme.probeController()
  nvme.shutdown()
  nvme.resetController()
  nvme.probeController()
  logger.info('PASS: Successfully identified the drive after controller reset')
}

function testPowerCycleUsingPowerOnAndOff () {
  options.logSubTest('testPowerCycleUsingPowerOnAndOff')

  powerController.init()
  nvme.probeController()
  powerController.powerOff()
  powerController.powerOff()
  logger.info('PASS: calling powerController.powerOn twice worked')

  powerController.powerOn()
  powerController.powerOn({ onHoldTimeMs: 30000 })
  logger.info('PASS: calling powerController.powerOn twice worked')

  nvme.probeController()
  logger.info('PASS: Successfully identified the drive after power on and off')
}

function testPowerCycleUsingPowerCycle () {
  options.logSubTest('testPowerCycleUsingPowerCycle')

  powerController.init()
  nvme.probeController()
  powerController.powerCycle({ onHoldTimeMs: 5000 })
  nvme.probeController()
  logger.info('PASS: Successfully identified the drive after power cycle')
}

function testPowerCycleWithShutdown () {
  options.logSubTest('testPowerCycleWithShutdown')

  powerController.init()
  nvme.probeController()
  nvme.shutdown()
  powerController.powerCycle({ onHoldTimeMs: 5000 })
  nvme.probeController()
  logger.info(
    'PASS: Successfully identified the drive after power cycle with shutdown'
  )
}

function testPowerCycleProbeController () {
  options.logSubTest('testPowerCycleProbeController')

  const unexpectedError = new Error('This should not be triggered')

  powerController.init()
  nvme.probeController()
  powerController.powerOff()
  try {
    nvme.probeController()
    throw unexpectedError
  } catch (e) {
    if (e === unexpectedError) {
      throw new Error('Did not receive an expected exception')
    }
  }
  logger.info(
    'PASS: Exception triggered when nvme.probeController is called after powerOff'
  )

  powerController.powerOn({ onHoldTimeMs: 5000 })
  nvme.probeController()
  logger.info(
    'PASS: nvme.probeController() completed successfully after powerOn'
  )

  nvme.probeController()
  logger.info(
    'PASS: nvme.probeController() completed successfully when called twice'
  )
}

function testPowerCycle () {
  let testOptions = options.get('powercycle') || { totalDuration: 60 }
  const totalDuration = testOptions.totalDuration
  const workloadGenDuration = Math.round(Math.max(totalDuration / 10, 30))
  testOptions = {
    ...testOptions,
    workloadGenerator: { duration: workloadGenDuration }
  }
  options.set('powercycle', testOptions)

  const testSuiteName = 'Power Cycle'
  options.logTestSuiteStart(testSuiteName, testOptions)

  // TODO: Randomize sub tests
  const subTests = [
    testPowerCycleProbeController,
    testPowerCycleWithShutdown,
    testPowerCycleUsingPowerCycle,
    testPowerCycleUsingPowerOnAndOff,
    testPowerCycleUsingWorkloadGenerator,
    testPowerCycleUsingWorkloadGeneratorAggresivePc,
    testShutdownWithReset
  ]

  for (const test of subTests) {
    test()
  }

  const t0 = Date.now()
  while (Date.now() - t0 < totalDuration * 1000) {
    subTests[Date.now() % subTests.length]()
  }

  options.logTestSuiteDone(testSuiteName)
}

testPowerCycle()
