importScript('/lib/nvme/index.js')
importScript('../misc/init.js')
;(function () {
  options.logSubTest('workload_basic_with_trim.js')

  nvme.probeController()
  logger.info('PASS: Succesfully connected to NVMe controller')

  const {
    loops = 1,
    duration = 10,
    getSmart = false
  } = options.get('workload') || {}

  nvme.probeController()

  const testConditions = [
    {
      nonZeroIoTypes: ['read', 'trim'],
      readPercent: 50,
      trimPercent: 50
    },
    {
      nonZeroIoTypes: ['write', 'trim'],
      readPercent: 0,
      trimPercent: 50
    },
    {
      nonZeroIoTypes: ['trim'],
      readPercent: 0,
      trimPercent: 100
    },
    {
      nonZeroIoTypes: ['read', 'write'],
      readPercent: 50,
      trimPercent: 0
    },
    {
      nonZeroIoTypes: ['read', 'write', 'trim'],
      readPercent: 30,
      trimPercent: 30
    }
  ]

  // Note: Samsung DCT 970 doesn't like Trim size that isn't multiples of 4k
  const xferSize = 4096

  for (const cond of testConditions) {
    const result = nvme.startWorkloadGenerator({
      nsid: 1,
      queueDepth: 32,
      xferSize,
      loops,
      readPercent: cond.readPercent,
      trimPercent: cond.trimPercent,
      workload: 'random',
      duration,
      features: {
        histogram: {
          enabled: true,
          startUs: 100,
          endUs: 100000,
          precisions: 4,
          rawData: true
        }
      }
    })

    const ioPercent = {
      read: cond.readPercent,
      trim: cond.trimPercent,
      write: 100 - cond.readPercent - cond.trimPercent
    }

    const availableIoTypes = ['read', 'write', 'trim']
    const nonZeroIoTypes = cond.nonZeroIoTypes
    const expectedPerfProps = ['bandwidth', 'iops', 'xfer', 'io']
    let totalIos = 0
    let totalXfers = 0
    for (const io of availableIoTypes) {
      const hasExpected = nonZeroIoTypes.includes(io)
        ? value => value !== 0
        : value => value === 0
      const message = nonZeroIoTypes.includes(io) ? 'non zero value' : '0'
      for (const prop of expectedPerfProps) {
        if (!hasExpected(result[io].perf[prop])) {
          throw new Error(
            `${io}.perf.${prop} from the result object should be ${message}`
          )
        }
      }
      totalIos += result[io].perf.io
      totalXfers += result[io].perf.xfer
      if (!result[io].latency) {
        throw new Error(`${io}.latency should exists from the result object`)
      }
      if (!result[io].histogramRawData) {
        throw new Error(
          `${io}.histogramRawData should exists from the result object`
        )
      }
    }

    for (const io of nonZeroIoTypes) {
      // Give 10% of tolerance
      const percent = ioPercent[io]
      const ios = result[io].perf.io
      const measuredIo = Math.round((ios / totalIos) * 10) * 10
      const xfers = result[io].perf.xfer
      const measuredXfer = Math.round((xfers / totalXfers) * 10) * 10
      if (percent !== measuredIo) {
        throw new Error(
          `Expected ${percent}% of ios from ${io}, but got ${measuredIo}%`
        )
      }
      if (percent !== measuredXfer) {
        throw new Error(
          `Expected ${percent}% of xfers from ${io}, but got ${measuredXfer}%`
        )
      }
    }

    if (getSmart) {
      nvme.getLogPageSmart()
    }
    logger.info(
      'PASS: Basic Workload Generator with ' +
        `trimPercent: ${cond.trimPercent}, readPercent: ${cond.readPercent}`
    )
  }
})()
