'use strict'

const { hostname, port } = utility.systemInfo()
const { taskId } = utility.getTaskInfo()

const syncManager = {}

syncManager._mountNfs = function (nfsAddr, nfsPath) {
  try {
    utility.exec(`mount ${nfsAddr} ${nfsPath} -o timeo=3,retrans=2,retry=0`)
  } catch (e) {
    logger.info(e.toString())
    return false
  }

  return true
}

syncManager._checkNfsPath = function (nfsTaskPath) {
  try {
    utility.exec(`ls ${nfsTaskPath}`)
    return true
  } catch (e) {
    return false
  }
}

syncManager.init = function (options) {
  const {
    nfsAddr = '',
    nfsPath = '',
    timeout = 300,
    wait = 60,
    period = 1000
  } = options || {}

  if (nfsAddr === '' || nfsPath === '') {
    throw new Error('NO NFS ADDR or PATH')
  }

  const env = {
    nfsAddr,
    nfsPath,
    nfsTaskPath: `${nfsPath}/task-${taskId}`,
    scriptPath: `/opt/qam/log/port${port}/task-${taskId}`
  }
  env.taskCounterPath = `${env.nfsTaskPath}/_counter`

  if (!syncManager._mountNfs(nfsAddr, nfsPath)) {
    throw new Error(`SYNC: FAILED MOUNT NFS ${nfsPath}`)
  } else {
    logger.info(`SYNC: MOUNT ${nfsPath}`)
  }

  if (!syncManager._checkNfsPath(env.nfsTaskPath)) {
    logger.info(`SYNC: MAKE A SYNC DIRECTORY ${env.nfsTaskPath}`)
    utility.exec(`mkdir -p ${env.nfsTaskPath}`)
  }

  syncManager._readNamedCounter = function (counterName) {
    try {
      const { stdout: res } = utility.exec(`flock ${env.taskCounterPath} node ${env.scriptPath}/sync_read.js ${counterName}`)
      return parseInt(res)
    } catch (e) {
      throw new Error(`SYNC: INVALID COUNTER NAME - ${counterName}`)
    }
  }

  syncManager._incrNamedCounter = function _incrNamedCounter (counterName) {
    const { stdout: res } = utility.exec(`flock ${env.taskCounterPath} node ${env.scriptPath}/sync_incr.js ${counterName}`)
    return parseInt(res)
  }
  syncManager.env = env

  const initScript = utility.readFile('sync_init.js', { baseDir: 'workspace' }).toString().replace('${syncCounterPath}', env.taskCounterPath)
  const incrScript = utility.readFile('sync_incr.js', { baseDir: 'workspace' }).toString().replace('${syncCounterPath}', env.taskCounterPath)
  const readScript = utility.readFile('sync_read.js', { baseDir: 'workspace' }).toString().replace('${syncCounterPath}', env.taskCounterPath)

  // copy sync script
  utility.createFile('sync_init.js', initScript)
  utility.createFile('sync_incr.js', incrScript)
  utility.createFile('sync_read.js', readScript)

  const startTime = ((new Date()) - 0) + (wait * 1000)
  while (true) {
    if (((new Date()) / 1000) - startTime > timeout) {
      throw new Error(`SYNC: INIT TIMEOUT ${timeout}s`)
    }

    const { stdout: res, stderr: err } = utility.exec(`flock ${env.taskCounterPath} node ${env.scriptPath}/sync_init.js ${startTime} ${period}`)

    if (err !== '') {
      logger.info(err)
      throw new Error('SYNC: FATAL ERROR - TIMER IS BROKEN')
    }

    const state = parseInt(res)
    switch (state) {
      case 0:
      case 1:
        logger.info('SYNC: INIT DONE')
        break
      case 2:
        logger.info('SYNC: WAIT')
        break
      case 3:
        logger.info('SYNC: WARNING - OVERWRITE OLD')
        break
      case 4:
        logger.info('SYNC: WARNING - FILE ERROR, OVERWRITE OLD')
        break
      default:
        throw new Error(`SYNC: FATAL ERROR - WRONG STATE: ${state}`)
    }

    if (state === 0 || state === 1) {
      break
    }
    logger.info('.')
    utility.sleep(period)
  }
}

syncManager.sync = function (syncPoint, func, options) {
  const { period = 1000, exclusiveCount = 0 } = options || {}
  const _readNamedCounter = syncManager._readNamedCounter
  const _incrNamedCounter = syncManager._incrNamedCounter

  function _isDone () {
    return _readNamedCounter(`${syncPoint}-in`) === _readNamedCounter(`${syncPoint}-out`)
  }

  try {
    const count = _incrNamedCounter(`${syncPoint}-in`)

    if (exclusiveCount === 0 || exclusiveCount >= count) {
      if (func) {
        func()
      }
    }
  } catch (e) {
    _incrNamedCounter(`${syncPoint}-err`)
    logger.info(e.toString())
    throw e
  } finally {
    _incrNamedCounter(`${syncPoint}-out`)
  }

  while (!_isDone()) {
    utility.sleep(period)
  }
}
