const serial = 'IOB-PORT2'

const product = 0x6011 // FT4232H
// const product = 0x6010 // FT2232H

logger.info('Opening interface A')
const idx = ftdi.open({
  vendor: 0x0403,
  product,
  interface: 1,
  serial
})

ftdi.setBaudRate(idx, 115200)
logger.create('ftdi', 'ftdi.log')
ftdi.pipe(idx, 'ftdi')

logger.info('Opening interface B')
const idx2 = ftdi.open({
  vendor: 0x0403,
  product,
  interface: 2,
  serial
})

let rpcId = 0

const STM32_COMMANDS = {
  INITIAL_SYNC: Buffer.from([0x7f]),
  GET: Buffer.from([0x00, 0xff]),
  ERASE: Buffer.from([0x43, 0xbc]),
  GLOBAL_ERASE: Buffer.from([0xff, 0x00]),
  WRITE_MEMORY: Buffer.from([0x31, 0xce]),
  ACK: 0x79,
  NACK: 0x1f
}

function getFirmwareVersion () {
  logger.info('Get Version')
  const id = sendRpcRequest({ method: 'getVersion', params: null })
  utility.sleep(3000)
  data = waitForRpcResponse(id)
  logger.info(data)
}

function enterMpsse () {
  logger.info('Entering MPSSE mode')
  ftdi.setBitmode(idx2, { bitmask: 0, mode: 2 })
}

function leaveMpsse () {
  logger.info('Leaving MPSSE mode')
  ftdi.setBitmode(idx2, { bitmask: 0, mode: 0 })
}

function waitForAck () {
  logger.info('Waiting for Ack')
  let buffer = Buffer.alloc(0)
  while (buffer.length != 1) {
    buffer = ftdi.read(idx, 1)
  }
  const byte = buffer[0]
  if (byte === 0x79) {
    logger.info('Received Ack')
  } else if (byte === 0x1f) {
    throw new Error('Received Nack')
  } else {
    throw new Error(logger.sprintf('Received Undefined 0x%02x', byte))
  }
}

function testPins () {
  enterMpsse()

  logger.info('Testing Pins')

  for (let i = 0; i < 10; i++) {
    logger.info('All BDBUS out high')
    ftdi.write(idx2, Buffer.from([0x80, 0xff, 0xff]))
    utility.sleep(5000)

    logger.info('All BDBUS out low')
    ftdi.write(idx2, Buffer.from([0x80, 0, 0xff]))
    utility.sleep(5000)
  }
}

function enterBootloader () {
  enterMpsse()

  logger.info('Entering into bootloader')

  const set_boot_pin = product === 0x6011 ? 0xef : 0xff
  const reset_assert_pin = product === 0x6011 ? 0xcf : 0xfb
  const reset_deassert_pin = product === 0x6011 ? 0xef : 0xff
  const gpio_addr = product === 0x6011 ? 0x80 : 0x82

  const set_boot0 = [gpio_addr, set_boot_pin, 0xff]
  const reset_assert = [gpio_addr, reset_assert_pin, 0xff]
  const reset_deassert = [gpio_addr, reset_deassert_pin, 0xff]

  logger.info(`Set Boot0: ${set_boot0}`)
  ftdi.write(idx2, Buffer.from(set_boot0))
  utility.sleep(1000)
  logger.info(`Asserting reset: ${reset_assert}`)
  ftdi.write(idx2, Buffer.from(reset_assert))
  utility.sleep(5000)
  logger.info(`Deasserting reset: ${reset_deassert}`)
  ftdi.write(idx2, Buffer.from(reset_deassert))
  utility.sleep(2000)

  ftdi.setParity(idx, 'even')
  ftdi.flushBuffer(idx)
  ftdi.useBinaryMode(idx)
  logger.info('Sending 0x7F to enter command')
  ftdi.write(idx, Buffer.from([0x7f]))
  waitForAck()

  leaveMpsse()
}

function enterFirmware () {
  enterMpsse()

  logger.info('Entering into Firmware')
  const assert_pin = product === 0x6011 ? 0xdf : 0xf9
  const deassrt_pin = product === 0x6011 ? 0xff : 0xfd
  const gpio_addr = product === 0x6011 ? 0x80 : 0x82

  const reset_assert = [gpio_addr, assert_pin, 0xff]
  const reset_deassert = [gpio_addr, deassrt_pin, 0xff]

  logger.info('Asserting reset')
  ftdi.write(idx2, Buffer.from(reset_assert))
  utility.sleep(5000)

  ftdi.flushBuffer(idx)
  ftdi.useTextMode(idx)
  ftdi.setParity(idx, 'none')

  logger.info('Deasserting reset')
  ftdi.write(idx2, Buffer.from(reset_deassert))
  utility.sleep(2000)

  leaveMpsse()
}

function sendBytes (bufferOrArray) {
  ftdi.write(idx, Buffer.from(bufferOrArray))
}

async function sendCommand (command) {
  sendBytes(command)
  return waitForAck()
}

function waitForBytes (numBytes) {
  logger.info(`Waiting for ${numBytes} bytes`)
  let buffer = Buffer.alloc(0)
  while (buffer.length != numBytes) {
    buffer = Buffer.concat([buffer, ftdi.read(idx, 1)])
  }
  return buffer
}

function waitForNewLine () {
  let buffer = Buffer.alloc(0)
  let lastByte = Buffer.alloc(0)
  while (lastByte.toString() !== '\n') {
    lastByte = ftdi.read(idx, 1)
    if (lastByte.length == 0) {
      continue
    }
    buffer = Buffer.concat([buffer, lastByte])
  }
  return buffer.toString()
}

function sendRpcRequest (data) {
  data.id = rpcId
  rpcId += 1
  command = `${JSON.stringify(data)}\n`
  logger.info(command)
  ftdi.write(idx, Buffer.alloc(command.length, command))
  return data.id
}

function waitForRpcResponse (id) {
  logger.info(`Waiting for RPC response. ID = ${id}`)
  response = null
  while (response === null) {
    data = waitForNewLine()
    logger.info(data)
    parsedData = JSON.parse(data)
    if (parsedData.id === id) {
      response = parsedData
    }
  }
  return response
}

function stmCommandGet () {
  logger.info('Sending STM Get command')
  const command = [0x00, 0xff]
  ftdi.write(idx, Buffer.from(command))
  waitForAck()
  let buffer = waitForBytes(1)
  const bytes = buffer.readUInt8() + 1
  logger.info('%d bytes will follow', bytes)
  buffer = waitForBytes(bytes)
  let bufferArray = [...buffer]
  bufferArray = bufferArray.map(item => logger.sprintf('0x%02X', item))
  logger.info(bufferArray)
  waitForAck()
}

function stmCommandGetVersion () {
  logger.info('Sending STM GetVersion command')
  const command = [0x01, 0xfe]
  ftdi.write(idx, Buffer.from(command))
  waitForAck()
  const bytes = waitForBytes(3)
  const bytesArray = [...bytes]
  const version = (bytesArray[0] / 10).toFixed(1)
  logger.info(`STM Bootloader Version: ${version}`)
  waitForAck()
}

function eraseMemory () {
  logger.info('Erasing Entire Flash')
  sendCommand(STM32_COMMANDS.ERASE)
  sendBytes(STM32_COMMANDS.GLOBAL_ERASE)
  waitForAck()
}

function getChecksum (buffer) {
  return buffer.reduce((acc, byte) => acc ^ byte, 0)
}

function writeMemory (address, data) {
  if (data.length > 256) {
    throw new Error('Data length exceeds maximum of 256 bytes.')
  }

  // Send the WRITE_MEMORY command.
  sendBytes(STM32_COMMANDS.WRITE_MEMORY)
  waitForAck() // Will throw an exception if ACK is not received.

  // Prepare the address and its checksum.
  const addressBuffer = Buffer.alloc(4)
  addressBuffer.writeUInt32BE(address) // Write address in Big Endian format.
  const addressChecksum = addressBuffer.reduce((acc, byte) => acc ^ byte, 0)

  // Send the address and its checksum.
  sendBytes(Buffer.concat([addressBuffer, Buffer.from([addressChecksum])]))
  waitForAck() // Will throw an exception if ACK is not received.

  // Prepare data length and its checksum.
  const dataSize = data.length - 1 // 0 < N ≤ 255 => N data bytes = dataSize + 1
  const dataChecksum = data.reduce((acc, byte) => acc ^ byte, dataSize)

  // Send the data length, data, and its checksum.
  sendBytes(
    Buffer.concat([Buffer.from([dataSize]), data, Buffer.from([dataChecksum])])
  )
  waitForAck() // Will throw an exception if ACK is not received.
}

function writeFirmware (bin) {
  const CHUNK_SIZE = 256 // Adjust as needed.
  let address = 0x08000000 // Starting address for STM32 flash memory

  logger.info(`Firmware size: ${bin.length}`)
  for (let offset = 0; offset < bin.length; offset += CHUNK_SIZE) {
    logger.info(`Writing Firmware Chunk at offset ${offset}`)
    const chunk = bin.slice(offset, offset + CHUNK_SIZE)
    writeMemory(address, chunk)
    address += CHUNK_SIZE
  }
  logger.info('Firmware programming complete!')
}

function readFirmwareBinary () {
  logger.info('Reading iob_pmu_firmware.bin')
  const bin = utility.readFile('iob_pmu_firmware.bin', { baseDir: 'workspace' })
  return bin
}

function programFirmware () {
  const bin = readFirmwareBinary()
  eraseMemory()
  writeFirmware(bin)
}

testPins()
// enterBootloader()
// stmCommandGet()
// stmCommandGetVersion()
// programFirmware()
// enterFirmware()
// getFirmwareVersion()

// getVersion()
// resetChip()
// getVersion()

utility.sleep(5000)
